<?php
/**
 * WebDAV服务器管理后台
 * 兼容PHP 7.3及以下版本
 */

// 开启会话
session_start();

// 加载配置
$config = require 'config.php';

// 检查是否已安装
if (!file_exists(__DIR__ . '/install.lock')) {
    header('Location: install.php');
    exit;
}

// 连接数据库
try {
    $dbPath = __DIR__ . '/' . $config['db_file'];
    $pdo = new PDO('sqlite:' . $dbPath);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    // 检查并添加access_dir字段
    try {
        $pdo->exec('ALTER TABLE users ADD COLUMN access_dir TEXT DEFAULT NULL');
    } catch (Exception $e) {
        // 字段已存在，忽略错误
    }
} catch (PDOException $e) {
    die('数据库连接失败：' . $e->getMessage());
}

// 处理登录
$loginError = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'login') {
    $username = isset($_POST['username']) ? trim($_POST['username']) : '';
    $password = isset($_POST['password']) ? trim($_POST['password']) : '';
    
    if (!empty($username) && !empty($password)) {
        $stmt = $pdo->prepare('SELECT * FROM users WHERE username = ? LIMIT 1');
        $stmt->execute([$username]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($user && password_verify($password, $user['password'])) {
            // 登录成功
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['username'] = $user['username'];
            $_SESSION['is_admin'] = $user['is_admin'];
            
            // 重定向到管理页面
            header('Location: admin.php');
            exit;
        } else {
            $loginError = '用户名或密码错误';
        }
    } else {
        $loginError = '请输入用户名和密码';
    }
}

// 处理退出登录 - 使用统一登出页面
if (isset($_GET['action']) && $_GET['action'] === 'logout') {
    header('Location: logout.php');
    exit;
}

// 检查是否已登录
$isLoggedIn = isset($_SESSION['user_id']);
$isAdmin = $isLoggedIn && isset($_SESSION['is_admin']) && $_SESSION['is_admin'] == 1;

// 处理用户管理操作
$userMessage = '';

// 从session中获取消息（用于PRG模式）
if (isset($_SESSION['user_message'])) {
    $userMessage = $_SESSION['user_message'];
    unset($_SESSION['user_message']);
}

// 只有管理员可以执行以下操作
if ($isAdmin) {
    // 添加用户
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'add_user') {
        $username = isset($_POST['username']) ? trim($_POST['username']) : '';
        $password = isset($_POST['password']) ? trim($_POST['password']) : '';
        $isAdminUser = 0; // 禁止创建新的管理员账号
        
        if (!empty($username) && !empty($password)) {
            try {
                // 检查用户名是否已存在
                $stmt = $pdo->prepare('SELECT COUNT(*) FROM users WHERE username = ?');
                $stmt->execute([$username]);
                $count = $stmt->fetchColumn();
                
                if ($count > 0) {
                    $_SESSION['user_message'] = '错误：用户名已存在';
                } else {
                    // 添加新用户
                    $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
                    $accessDir = isset($_POST['access_dir']) ? trim($_POST['access_dir']) : null;
                    if (empty($accessDir)) {
                        $accessDir = $username;
                    }
                    
                    $stmt = $pdo->prepare('INSERT INTO users (username, password, is_admin, access_dir) VALUES (?, ?, ?, ?)');
                    $stmt->execute([$username, $hashedPassword, $isAdminUser, $accessDir]);
                    // 创建用户目录
                    $userDir = $accessDir ?: $username;
                    $userPath = __DIR__ . '/webdav/' . $userDir;
                    if (!is_dir($userPath)) {
                        mkdir($userPath, 0755, true);
                    }
                    $_SESSION['user_message'] = '用户添加成功';
                }
            } catch (Exception $e) {
                $_SESSION['user_message'] = '错误：' . $e->getMessage();
            }
        } else {
            $_SESSION['user_message'] = '错误：用户名和密码不能为空';
        }
        
        // 重定向到用户管理页面
        $redirectUrl = 'admin.php#users';
        header('Location: ' . $redirectUrl);
        exit;
    }
    
    // 删除用户
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'delete_user') {
        $userId = isset($_POST['user_id']) ? (int)$_POST['user_id'] : 0;
        
        if ($userId > 0) {
            try {
                // 检查是否是管理员账号
                $stmt = $pdo->prepare('SELECT is_admin FROM users WHERE id = ?');
                $stmt->execute([$userId]);
                $isAdmin = $stmt->fetchColumn();
                
                if ($isAdmin) {
                    $_SESSION['user_message'] = '错误：不能删除管理员账号';
                } else {
                    $stmt = $pdo->prepare('DELETE FROM users WHERE id = ?');
                    $stmt->execute([$userId]);
                    $_SESSION['user_message'] = '用户删除成功';
                }
            } catch (Exception $e) {
                $_SESSION['user_message'] = '错误：' . $e->getMessage();
            }
        }
        
        // 重定向到用户管理页面
        $redirectUrl = 'admin.php#users';
        header('Location: ' . $redirectUrl);
        exit;
    }
    
    // 修改用户密码
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'change_password') {
        $userId = isset($_POST['user_id']) ? (int)$_POST['user_id'] : 0;
        $newPassword = isset($_POST['new_password']) ? trim($_POST['new_password']) : '';
        
        if ($userId > 0 && !empty($newPassword)) {
            try {
                $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
                $stmt = $pdo->prepare('UPDATE users SET password = ? WHERE id = ?');
                $stmt->execute([$hashedPassword, $userId]);
                $_SESSION['user_message'] = '密码修改成功';
            } catch (Exception $e) {
                $_SESSION['user_message'] = '错误：' . $e->getMessage();
            }
        } else {
            $_SESSION['user_message'] = '错误：新密码不能为空';
        }
        
        // 重定向到用户管理页面
        $redirectUrl = 'admin.php#users';
        header('Location: ' . $redirectUrl);
        exit;
    }

    // 修改用户访问目录
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'change_access_dir') {
        $userId = isset($_POST['user_id']) ? (int)$_POST['user_id'] : 0;
        $accessDir = isset($_POST['access_dir']) ? trim($_POST['access_dir']) : '';
        
        if ($userId > 0) {
            try {
                // 如果访问目录为空，使用用户名
                if (empty($accessDir)) {
                    $stmt = $pdo->prepare('SELECT username FROM users WHERE id = ?');
                    $stmt->execute([$userId]);
                    $user = $stmt->fetch(PDO::FETCH_ASSOC);
                    $accessDir = $user['username'];
                }
                
                $stmt = $pdo->prepare('UPDATE users SET access_dir = ? WHERE id = ? AND is_admin = 0');
                $stmt->execute([$accessDir, $userId]);
                $_SESSION['user_message'] = '访问目录修改成功';
            } catch (Exception $e) {
                $_SESSION['user_message'] = '错误：' . $e->getMessage();
            }
        } else {
            $_SESSION['user_message'] = '错误：用户ID无效';
        }
        
        // 重定向到用户管理页面
        $redirectUrl = 'admin.php#users';
        header('Location: ' . $redirectUrl);
        exit;
    }
}

// 获取所有用户列表（仅管理员可见）
$users = [];
if ($isAdmin) {
    // 确保只有一个管理员账号
    $stmt = $pdo->query('SELECT COUNT(*) FROM users WHERE is_admin = 1');
    $adminCount = $stmt->fetchColumn();
    
    if ($adminCount > 1) {
        // 如果存在多个管理员，只保留最早创建的那个
        $stmt = $pdo->prepare('SELECT id FROM users WHERE is_admin = 1 ORDER BY id ASC LIMIT 1');
        $stmt->execute();
        $firstAdmin = $stmt->fetchColumn();
        
        // 将其他管理员降级为普通用户
        $stmt = $pdo->prepare('UPDATE users SET is_admin = 0 WHERE is_admin = 1 AND id != ?');
        $stmt->execute([$firstAdmin]);
    }
    
    // 分页参数
    $page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
    $perPage = isset($_GET['per_page']) ? max(1, intval($_GET['per_page'])) : 8;
    
    // 获取总记录数
    $countStmt = $pdo->query('SELECT COUNT(*) FROM users WHERE is_admin = 0');
    $totalUsers = $countStmt->fetchColumn();
    
    // 计算总页数
    $totalPages = max(1, ceil($totalUsers / $perPage));
    
    // 限制页码范围
    $page = min($page, $totalPages);
    
    // 计算偏移量
    $offset = ($page - 1) * $perPage;
    
    // 获取分页数据
    $stmt = $pdo->prepare('SELECT * FROM users WHERE is_admin = 0 ORDER BY created_at DESC LIMIT ? OFFSET ?');
    $stmt->bindValue(1, $perPage, PDO::PARAM_INT);
    $stmt->bindValue(2, $offset, PDO::PARAM_INT);
    $stmt->execute();
    $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// 分页渲染函数
function renderPagination($currentPage, $totalPages, $perPage, $totalUsers) {
    $baseUrl = strtok($_SERVER['REQUEST_URI'], '?');
    $queryParams = $_GET;
    
    echo '<div class="pagination-container">';
    echo '<div class="pagination-info">';
    echo '显示 ' . (($currentPage - 1) * $perPage + 1) . ' - ' . min($currentPage * $perPage, $totalUsers) . ' 条，共 ' . $totalUsers . ' 条';
    echo '</div>';
    
    echo '<div class="pagination-select">';
    echo '<select onchange="changePerPage(this.value)">';
    $options = [8, 20, 50, 100];
    foreach ($options as $option) {
        $selected = $option == $perPage ? 'selected' : '';
        echo "<option value=\"$option\" $selected>每页 $option 条</option>";
    }
    echo '</select>';
    echo '</div>';
    
    echo '<div class="pagination">';
    
    // 上一页
    if ($currentPage > 1) {
        echo '<button class="pagination-btn" onclick="goToPage(' . ($currentPage - 1) . ')">上一页</button>';
    } else {
        echo '<button class="pagination-btn" disabled>上一页</button>';
    }
    
    // 页码
    $startPage = max(1, $currentPage - 2);
    $endPage = min($totalPages, $currentPage + 2);
    
    if ($startPage > 1) {
        echo '<button class="pagination-btn" onclick="goToPage(1)">1</button>';
        if ($startPage > 2) {
            echo '<span class="pagination-dots">...</span>';
        }
    }
    
    for ($i = $startPage; $i <= $endPage; $i++) {
        $activeClass = $i == $currentPage ? 'active' : '';
        echo '<button class="pagination-btn ' . $activeClass . '" onclick="goToPage(' . $i . ')">' . $i . '</button>';
    }
    
    if ($endPage < $totalPages) {
        if ($endPage < $totalPages - 1) {
            echo '<span class="pagination-dots">...</span>';
        }
        echo '<button class="pagination-btn" onclick="goToPage(' . $totalPages . ')">' . $totalPages . '</button>';
    }
    
    // 下一页
    if ($currentPage < $totalPages) {
        echo '<button class="pagination-btn" onclick="goToPage(' . ($currentPage + 1) . ')">下一页</button>';
    } else {
        echo '<button class="pagination-btn" disabled>下一页</button>';
    }
    
    echo '</div>';
    echo '</div>';
}

// 处理个人信息修改
$profileMessage = '';

// 从session中获取个人信息消息（用于PRG模式）
if (isset($_SESSION['profile_message'])) {
    $profileMessage = $_SESSION['profile_message'];
    unset($_SESSION['profile_message']);
}

// 修改当前用户密码
if ($isLoggedIn && $_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'change_own_password') {
    $currentPassword = isset($_POST['current_password']) ? trim($_POST['current_password']) : '';
    $newPassword = isset($_POST['new_password']) ? trim($_POST['new_password']) : '';
    $confirmPassword = isset($_POST['confirm_password']) ? trim($_POST['confirm_password']) : '';
    
    if (!empty($currentPassword) && !empty($newPassword) && !empty($confirmPassword)) {
        if ($newPassword !== $confirmPassword) {
            $_SESSION['profile_message'] = '错误：两次输入的新密码不一致';
        } else {
            // 验证当前密码
            $stmt = $pdo->prepare('SELECT * FROM users WHERE id = ? LIMIT 1');
            $stmt->execute([$_SESSION['user_id']]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($user && password_verify($currentPassword, $user['password'])) {
                // 更新密码
                $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
                $stmt = $pdo->prepare('UPDATE users SET password = ? WHERE id = ?');
                $stmt->execute([$hashedPassword, $_SESSION['user_id']]);
                $_SESSION['profile_message'] = '密码修改成功';
            } else {
                $_SESSION['profile_message'] = '错误：当前密码不正确';
            }
        }
    } else {
        $_SESSION['profile_message'] = '错误：所有密码字段都必须填写';
    }
    
    // 重定向到账户管理页面
    $redirectUrl = 'admin.php#account';
    header('Location: ' . $redirectUrl);
    exit;
}

// 修改当前用户名
if ($isLoggedIn && $_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'change_username') {
    $newUsername = isset($_POST['new_username']) ? trim($_POST['new_username']) : '';
    $password = isset($_POST['password']) ? trim($_POST['password']) : '';
    
    if (!empty($newUsername) && !empty($password)) {
        // 验证密码
        $stmt = $pdo->prepare('SELECT * FROM users WHERE id = ? LIMIT 1');
        $stmt->execute([$_SESSION['user_id']]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($user && password_verify($password, $user['password'])) {
            // 检查新用户名是否已存在
            $stmt = $pdo->prepare('SELECT COUNT(*) FROM users WHERE username = ? AND id != ?');
            $stmt->execute([$newUsername, $_SESSION['user_id']]);
            $count = $stmt->fetchColumn();
            
            if ($count > 0) {
                $_SESSION['profile_message'] = '错误：用户名已存在';
            } else {
                // 更新用户名
                $stmt = $pdo->prepare('UPDATE users SET username = ? WHERE id = ?');
                $stmt->execute([$newUsername, $_SESSION['user_id']]);
                $_SESSION['username'] = $newUsername;
                $_SESSION['profile_message'] = '用户名修改成功';
            }
        } else {
            $_SESSION['profile_message'] = '错误：密码不正确';
        }
    } else {
        $_SESSION['profile_message'] = '错误：新用户名和密码不能为空';
    }
    
    // 重定向到账户管理页面
    $redirectUrl = 'admin.php#account';
    header('Location: ' . $redirectUrl);
    exit;
}

// HTML输出
?>
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>WebDAV服务器管理后台</title>
        <style>
            /* 基础样式 */
            * {
                box-sizing: border-box;
            }
            
            body {
                font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Arial, sans-serif;
                line-height: 1.6;
                margin: 0;
                padding: 10px;
                background-color: #f5f5f5;
                font-size: 16px;
            }
            
            /* 响应式样式：隐藏移动设备上的创建时间列 */
            @media screen and (max-width: 768px) {
                .hide-on-mobile {
                    display: none;
                }
                
                /* 调整表格在移动设备上的显示 */
                table {
                    font-size: 14px;
                }
                
                td, th {
                    padding: 8px 4px;
                }
                
                /* 确保操作按钮区域足够空间 */
                td:last-child {
                    min-width: 120px;
                }
                
                /* 分页组件在小屏幕上保持水平布局，但允许换行 */
                .pagination-container {
                    flex-direction: row;
                    align-items: center;
                    justify-content: flex-start;
                    text-align: left;
                }
                
                .pagination-info,
                .pagination-select,
                .pagination {
                    flex: 0 0 auto;
                    text-align: left;
                    margin-bottom: 5px;
                }
                
                .pagination-info {
                    margin-right: 10px;
                }
                
                .pagination-select {
                    margin-left: 0;
                    margin-right: 10px;
                }
                
                .pagination {
                    flex: 1 1 100%;
                    text-align: left;
                    margin-top: 10px;
                }
            }
            
            /* 帮助弹窗样式 */
            .help-modal {
                display: none;
                position: fixed;
                z-index: 1000;
                left: 0;
                top: 0;
                width: 100%;
                height: 100%;
                background-color: rgba(0,0,0,0.4);
            }
            
            .help-modal-content {
                background-color: #fefefe;
                margin: 10% auto;
                padding: 20px;
                border: 1px solid #888;
                border-radius: 8px;
                width: 80%;
                max-width: 600px;
                max-height: 80vh;
                overflow-y: auto;
            }
            
            .help-modal-close {
                color: #aaa;
                float: right;
                font-size: 28px;
                font-weight: bold;
                cursor: pointer;
            }
            
            .help-modal-close:hover {
                color: #000;
            }
            
            .help-icon:hover svg {
                opacity: 0.8;
            }
            
            .help-content h3 {
                color: #2c3e50;
                margin-top: 15px;
                margin-bottom: 10px;
            }
            
            .help-content p {
                line-height: 1.6;
                margin-bottom: 10px;
            }
            
            .help-content .warning {
                background-color: #fff3cd;
                border: 1px solid #ffeaa7;
                border-radius: 4px;
                padding: 10px;
                margin: 10px 0;
            }
        
        .container {
            max-width: 1000px;
            margin: 0 auto;
            background-color: #fff;
            padding: 15px;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        
        /* 分页样式 */
        .pagination {
            display: flex;
            justify-content: center;
            align-items: center;
            margin: 30px 0;
            gap: 8px;
            flex-wrap: wrap;
        }
        
        .pagination-info {
            color: #666;
            margin: 0 15px;
            font-size: 14px;
        }
        
        .pagination-select {
            padding: 6px 10px;
            border: 1px solid #ddd;
            border-radius: 4px;
            font-size: 14px;
            background-color: #fff;
            cursor: pointer;
        }
        
        .pagination-btn {
            padding: 8px 12px;
            border: 1px solid #ddd;
            background-color: #fff;
            color: #333;
            text-decoration: none;
            border-radius: 4px;
            font-size: 14px;
            transition: all 0.2s ease;
            cursor: pointer;
        }
        
        .pagination-btn:hover {
            background-color: #f8f9fa;
            border-color: #007bff;
            color: #007bff;
        }
        
        .pagination-btn.active {
            background-color: #007bff;
            border-color: #007bff;
            color: white;
        }
        
        .pagination-btn.disabled {
            background-color: #f8f9fa;
            border-color: #e9ecef;
            color: #6c757d;
            cursor: not-allowed;
            pointer-events: none;
        }
        
        .pagination-ellipsis {
            padding: 8px 12px;
            color: #6c757d;
            font-size: 14px;
        }
        .pagination-container {
            display: flex;
            flex-direction: row;
            align-items: center;
            justify-content: flex-start;
            gap: 15px;
            margin: 20px 0;
            flex-wrap: wrap;
        }
        
        .pagination-info {
            flex: 0 0 auto;
            text-align: left;
            white-space: nowrap;
        }
        
        .pagination-select {
            flex: 0 0 auto;
            margin-left: 10px;
        }
        
        .pagination {
            flex: 1;
            text-align: right;
            white-space: nowrap;
        }
        
        h1 {
            color: #333;
            border-bottom: 1px solid #eee;
            padding-bottom: 10px;
            font-size: 24px;
            margin: 0 0 20px 0;
        }
        
        h2 {
            color: #333;
            border-bottom: 1px solid #eee;
            padding-bottom: 8px;
            font-size: 20px;
            margin: 25px 0 15px 0;
        }
        
        .message {
            padding: 12px;
            border-radius: 4px;
            margin-bottom: 15px;
            font-size: 14px;
            word-wrap: break-word;
        }
        
        .error {
            color: #e74c3c;
            background-color: #fadbd8;
            border: 1px solid #e74c3c;
        }
        
        .success {
            color: #27ae60;
            background-color: #d4efdf;
            border: 1px solid #27ae60;
        }
        
        label {
            display: block;
            margin-bottom: 5px;
            font-weight: 600;
            font-size: 14px;
            color: #555;
        }
        
        input[type="text"],
        input[type="password"] {
            width: 100%;
            padding: 10px;
            margin-bottom: 12px;
            border: 1px solid #ddd;
            border-radius: 4px;
            font-size: 16px;
            transition: border-color 0.3s;
        }
        
        input[type="text"]:focus,
        input[type="password"]:focus {
            outline: none;
            border-color: #3498db;
        }
        
        button, .button {
            background-color: #3498db;
            color: white;
            border: none;
            padding: 10px 16px;
            border-radius: 4px;
            cursor: pointer;
            text-decoration: none;
            display: inline-block;
            font-size: 14px;
            transition: background-color 0.3s;
            margin: 2px;
        }
        
        button:hover, .button:hover {
            background-color: #2980b9;
        }
        
        .button-danger {
            background-color: #e74c3c;
        }
        
        .button-danger:hover {
            background-color: #c0392b;
        }
        
        .icon-button {
            background: none;
            border: none;
            cursor: pointer;
            padding: 8px;
            transition: all 0.2s ease;
            border-radius: 4px;
            display: inline-flex;
            align-items: center;
            justify-content: center;
        }
        
        .icon-button:hover {
            background-color: rgba(0, 0, 0, 0.1);
            transform: scale(1.1);
        }
        
        .icon-button svg {
            width: 20px;
            height: 20px;
            fill: currentColor;
        }
        
        .icon-button.modify {
            color: #3498db;
        }
        
        .icon-button.delete {
            color: #e74c3c;
        }
        
        .icon-button:hover.modify {
            background-color: rgba(52, 152, 219, 0.1);
            color: #2980b9;
        }
        
        .icon-button:hover.delete {
            background-color: rgba(231, 76, 60, 0.1);
            color: #c0392b;
        }
        
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            flex-wrap: wrap;
            gap: 10px;
        }
        
        .user-info {
            text-align: right;
            font-size: 14px;
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
            margin-bottom: 20px;
            font-size: 14px;
        }
        
        table, th, td {
            border: 1px solid #ddd;
        }
        
        th, td {
            padding: 8px;
            text-align: left;
            word-break: break-word;
        }
        
        th {
            background-color: #f8f9fa;
            font-weight: 600;
            color: #495057;
        }
        
        /* WebDAV信息表格特殊样式 */
        #webdav table {
            border: 1px solid #ddd;
        }
        
        #webdav th,
        #webdav td {
            border: 1px solid #ddd;
        }
        
        .form-group {
            margin-bottom: 15px;
        }
        
        .tabs {
            display: flex;
            margin-bottom: 20px;
            border-bottom: 1px solid #ddd;
            flex-wrap: wrap;
            gap: 2px;
        }
        
        .tab {
            padding: 10px 15px;
            cursor: pointer;
            border: 1px solid #ddd;
            border-bottom: none;
            border-radius: 4px 4px 0 0;
            background-color: #f8f9fa;
            transition: all 0.3s;
            font-size: 14px;
            white-space: nowrap;
        }
        
        .tab.active {
            background-color: #fff;
            border-bottom: 1px solid #fff;
            margin-bottom: -1px;
            color: #3498db;
            font-weight: 600;
        }
        
        .tab:hover {
            background-color: #e9ecef;
        }
        
        .tab-content {
            display: none;
        }
        
        .tab-content.active {
            display: block;
        }
        
        /* 响应式设计 */
        @media (max-width: 768px) {
                body {
                    padding: 5px;
                    font-size: 14px;
                }
                
                .container {
                    padding: 12px;
                    border-radius: 0;
                    box-shadow: none;
                    margin: 0;
                }
                
                h1 {
                    font-size: 20px;
                    margin-bottom: 15px;
                }
                
                h2 {
                    font-size: 18px;
                    margin: 20px 0 12px 0;
                }
                
                .header {
                    flex-direction: column;
                    align-items: flex-start;
                    gap: 8px;
                }
                
                .user-info {
                    text-align: left;
                    font-size: 13px;
                }
                
                .tabs {
                    flex-direction: row;
                    overflow-x: auto;
                    -webkit-overflow-scrolling: touch;
                    gap: 1px;
                }
                
                .tab {
                    padding: 8px 12px;
                    font-size: 13px;
                    flex-shrink: 0;
                }
                
                table {
                    font-size: 12px;
                    display: table;
                    width: 100%;
                    table-layout: fixed;
                    border-radius: 4px;
                }
                
                th, td {
                    padding: 6px 8px;
                    min-width: 60px;
                }
                
                input[type="text"],
                input[type="password"] {
                    font-size: 16px; /* 防止iOS缩放 */
                    padding: 12px;
                }
                
                button, .button {
                    font-size: 14px;
                    padding: 12px 16px;
                    width: 100%;
                    margin: 4px 0;
                    text-align: center;
                }
                
                .message {
                    font-size: 13px;
                    padding: 10px;
                }
                
                #webdav-url-link {
                    font-size: 12px;
                    word-break: break-all;
                }
                
                #webdav td:last-child {
                    white-space: normal;
                }
                
                .icon-button {
                    padding: 8px;
                }
                
                .icon-button svg {
                    width: 20px;
                    height: 20px;
                }
                
                td > div[style*="flex"] {
                    flex-direction: row;
                    flex-wrap: nowrap;
                    gap: 6px;
                    justify-content: flex-start;
                }
                
                /* 平板模式下简化WebDAV表格边框 */
                #webdav table {
                    border: 1px solid #e0e0e0;
                    box-shadow: 0 1px 3px rgba(0,0,0,0.1);
                }
                
                #webdav th,
                #webdav td {
                    border: 1px solid #e0e0e0;
                    padding: 10px 8px;
                }
                
                @media (max-width: 480px) {
                    .icon-button {
                        padding: 8px;
                    }
                    
                    .icon-button svg {
                        width: 18px;
                        height: 18px;
                    }
                    
                    td > div[style*="flex"] {
                        gap: 4px;
                    }
                }
            }
        
        @media (max-width: 480px) {
            .container {
                padding: 10px;
            }
            
            h1 {
                font-size: 18px;
            }
            
            h2 {
                font-size: 16px;
            }
            
            .tab {
                padding: 6px 10px;
                font-size: 12px;
            }
            
            table {
                font-size: 14px;
                width: 100%;
                min-width: 100%;
                display: table;
                table-layout: fixed;
                border-collapse: collapse;
            }
            
            th, td {
                padding: 12px 8px;
                font-size: 16px;
                line-height: 1.5;
                word-break: break-word;
            }
            
            .user-info {
                font-size: 12px;
            }
            
            #webdav-url-link {
                font-size: 11px;
            }
            
            /* 手机模式下WebDAV表格优化 */
            #webdav table {
                width: 100%;
                min-width: 100%;
                display: table;
                table-layout: fixed;
                border: none;
                box-shadow: none;
                margin: 0;
            }
            
            #webdav th,
            #webdav td {
                border: none;
                border-bottom: 1px solid #eee;
                padding: 15px 12px;
                font-size: 16px;
                line-height: 1.6;
                word-break: break-word;
                vertical-align: top;
            }
            
            #webdav th {
                background-color: transparent;
                font-weight: 600;
                color: #333;
                width: 35%;
                min-width: 100px;
            }
            
            #webdav td {
                width: 65%;
                color: #555;
            }
            
            #webdav tr:last-child th,
            #webdav tr:last-child td {
                border-bottom: none;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <?php if (!$isLoggedIn): ?>
            <!-- 登录表单 -->
            <h1>WebDAV服务器管理后台</h1>
            
            <?php if (!empty($loginError)): ?>
                <div class="message error"><?php echo htmlspecialchars($loginError); ?></div>
            <?php endif; ?>
            
            <form method="post" action="">
                <input type="hidden" name="action" value="login">
                
                <div class="form-group">
                    <label for="username">用户名：</label>
                    <input type="text" id="username" name="username" required>
                </div>
                
                <div class="form-group">
                    <label for="password">密码：</label>
                    <input type="password" id="password" name="password" required>
                </div>
                
                <div class="form-group">
                    <button type="submit">登录</button>
                </div>
            </form>
            
        <?php else: ?>
            <!-- 管理后台 -->
            <div class="header">
                <h1>WebDAV服务器管理后台</h1>
                <div class="user-info">
                    欢迎，<?php echo htmlspecialchars($_SESSION['username']); ?> | 
                    <a href="admin.php?action=logout" style="color: #e74c3c; text-decoration: none; font-weight: 600;">退出</a>
                </div>
            </div>
            
            <?php if (!empty($userMessage)): ?>
                <div class="message <?php echo strpos($userMessage, '错误') === 0 ? 'error' : 'success'; ?>">
                    <?php echo htmlspecialchars($userMessage); ?>
                </div>
            <?php endif; ?>
            
            <div class="tabs">
                <div class="tab active" data-tab="account">账号设置</div>
                <?php if ($isAdmin): ?>
                    <div class="tab" data-tab="users">用户管理</div>
                <?php endif; ?>
                <div class="tab" data-tab="webdav">WebDAV信息</div>
            </div>
            
            <!-- 账号设置 -->
            <div id="account" class="tab-content active">
                <h2>修改用户名</h2>
                <form method="post" action="">
                    <input type="hidden" name="action" value="change_username">
                    
                    <div class="form-group">
                        <label for="new_username">新用户名：</label>
                        <input type="text" id="new_username" name="new_username" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="password_confirm">当前密码（确认）：</label>
                        <input type="password" id="password_confirm" name="password" required>
                    </div>
                    
                    <div class="form-group">
                        <button type="submit">修改用户名</button>
                    </div>
                </form>
                
                <h2>修改密码</h2>
                <form method="post" action="">
                    <input type="hidden" name="action" value="change_own_password">
                    
                    <div class="form-group">
                        <label for="current_password">当前密码：</label>
                        <input type="password" id="current_password" name="current_password" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="new_password">新密码：</label>
                        <input type="password" id="new_password" name="new_password" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="confirm_password">确认新密码：</label>
                        <input type="password" id="confirm_password" name="confirm_password" required>
                    </div>
                    
                    <div class="form-group">
                        <button type="submit">修改密码</button>
                    </div>
                </form>
            </div>
            
            <?php if ($isAdmin): ?>
                <!-- 用户管理 -->
                <div id="users" class="tab-content">
                    <h2>用户列表</h2>
                    <table>
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>用户名</th>
                                <th>访问目录</th>
                                <th class="hide-on-mobile">创建时间</th>
                                <th>操作</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($users as $user): ?>
                                <tr>
                                    <td><?php echo $user['id']; ?></td>
                                    <td><?php echo htmlspecialchars($user['username']); ?></td>
                                    <td><?php echo htmlspecialchars($user['access_dir'] ?: $user['username']); ?></td>
                                    <td class="hide-on-mobile"><?php echo $user['created_at']; ?></td>
                                <td>
                                        <div style="display: flex; gap: 8px; align-items: center;">
                                            <?php if (!$user['is_admin']): ?>
                                                <button type="button" class="icon-button modify" title="修改访问目录" 
                                                        onclick="showChangeAccessDirForm(<?php echo $user['id']; ?>, '<?php echo htmlspecialchars($user['username']); ?>', '<?php echo htmlspecialchars($user['access_dir'] ?: $user['username']); ?>')">
                                                    <svg viewBox="0 0 24 24">
                                                        <path d="M10 4H4c-1.1 0-1.99.9-1.99 2L2 18c0 1.1.9 2 2 2h16c1.1 0 2-.9 2-2V8c0-1.1-.9-2-2-2h-8l-2-2z"/>
                                                    </svg>
                                                </button>
                                                <button type="button" class="icon-button modify" title="修改密码" 
                                                        onclick="showChangePasswordForm(<?php echo $user['id']; ?>, '<?php echo htmlspecialchars($user['username']); ?>')">
                                                    <svg viewBox="0 0 24 24">
                                                        <path d="M20.71 7.04c.39-.39.39-1.04 0-1.41l-2.34-2.34c-.37-.39-1.02-.39-1.41 0l-1.84 1.83 3.75 3.75M3 17.25V21h3.75L17.81 9.94l-3.75-3.75L3 17.25z"/>
                                                    </svg>
                                                </button>
                                            <?php endif; ?>
                                            <?php if ($user['id'] != $_SESSION['user_id'] && !$user['is_admin']): ?>
                                                <form method="post" action="" style="display:inline;" onsubmit="return confirm('确定要删除用户 <?php echo htmlspecialchars($user['username']); ?> 吗？用户访问目录及其文件不会删除')">
                                                    <input type="hidden" name="action" value="delete_user">
                                                    <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                                                    <button type="submit" class="icon-button delete" title="删除用户">
                                                        <svg viewBox="0 0 24 24">
                                                            <path d="M19 4h-3.5l-1-1h-5l-1 1H5v2h14V4zM6 7v12c0 1.1.9 2 2 2h8c1.1 0 2-.9 2-2V7H6zm7 10.5c0 .3-.2.5-.5.5s-.5-.2-.5-.5v-6c0-.3.2-.5.5-.5s.5.2.5.5v6zm-3 0c0 .3-.2.5-.5.5s-.5-.2-.5-.5v-6c0-.3.2-.5.5-.5s.5.2.5.5v6zm5 0c0 .3-.2.5-.5.5s-.5-.2-.5-.5v-6c0-.3.2-.5.5-.5s.5.2.5.5v6z"/>
                                                        </svg>
                                                    </button>
                                                </form>
                                            <?php endif; ?>
                                        </div>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                    
                    <?php if ($totalPages > 1): ?>
                        <?php renderPagination($page, $totalPages, $perPage, $totalUsers); ?>
                    <?php endif; ?>
                    
                    <h2>添加新用户
                        <span class="help-icon" onclick="showAddUserHelp()" title="点击了解用户创建说明">
                            <svg viewBox="0 0 24 24" style="width: 20px; height: 20px; vertical-align: middle; margin-left: 8px; cursor: pointer;">
                                <circle cx="12" cy="12" r="10" fill="#3498db"/>
                                <text x="12" y="16" text-anchor="middle" fill="white" font-size="14" font-weight="bold">?</text>
                            </svg>
                        </span>
                    </h2>
                    <form method="post" action="">
                        <input type="hidden" name="action" value="add_user">
                        
                        <div class="form-group">
                            <label for="add_username">用户名：</label>
                            <input type="text" id="add_username" name="username" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="add_password">密码：</label>
                            <div style="display: flex; align-items: center; gap: 10px;">
                                <input type="password" id="add_password" name="password" required style="flex: 1;">
                                <label style="display: flex; align-items: center; gap: 5px; white-space: nowrap; font-size: 14px; cursor: pointer;" title="用于使用同一密码批量添加用户，使用完请取消勾选">
                                    <input type="checkbox" id="save_password" onchange="toggleSavePassword()">
                                    保存密码
                                </label>
                            </div>
                        </div>

                        <div class="form-group">
                            <label for="access_dir">限制访问目录：</label>
                            <input type="text" id="access_dir" name="access_dir" 
                                   placeholder="留空则使用用户名作为目录名"
                                   title="用户通过WebDAV访问时只能看到此目录下的文件">
                        </div>
                        
                        <div class="form-group">
                            <button type="submit">添加用户</button>
                        </div>
                    </form>
                    
                    <!-- 修改密码表单（隐藏） -->
                    <div id="change_password_form" style="display:none;">
                        <h2>修改用户密码</h2>
                        <form method="post" action="">
                            <input type="hidden" name="action" value="change_password">
                            <input type="hidden" id="change_user_id" name="user_id" value="">
                            
                            <div class="form-group">
                                <label for="change_username">用户名：</label>
                                <input type="text" id="change_username" disabled>
                            </div>
                            
                            <div class="form-group">
                                <label for="change_new_password">新密码：</label>
                                <input type="password" id="change_new_password" name="new_password" required>
                            </div>
                            
                            <div class="form-group">
                                <button type="submit">修改密码</button>
                                <button type="button" onclick="hideChangePasswordForm()">取消</button>
                            </div>
                        </form>
                    </div>

                    <!-- 修改访问目录表单（隐藏） -->
                    <div id="change_access_dir_form" style="display:none;">
                        <h2>修改访问目录</h2>
                        <form method="post" action="" onsubmit="return confirmChangeAccessDir()">
                            <input type="hidden" name="action" value="change_access_dir">
                            <input type="hidden" id="change_access_dir_user_id" name="user_id" value="">
                            
                            <div class="form-group">
                                <label for="change_access_dir_username">用户名：</label>
                                <input type="text" id="change_access_dir_username" disabled>
                            </div>
                            
                            <div class="form-group">
                                <label for="change_access_dir">访问目录：</label>
                                <input type="text" id="change_access_dir" name="access_dir" required
                                       placeholder="留空则使用用户名作为目录名"
                                       title="用户通过WebDAV访问时只能看到此目录下的文件">
                            </div>
                            
                            <div class="form-group">
                                <button type="submit">修改目录</button>
                                <button type="button" onclick="hideChangeAccessDirForm()">取消</button>
                            </div>
                        </form>
                    </div>
                </div>
            <?php endif; ?>
            
            <!-- 添加用户帮助弹窗 -->
            <div id="addUserHelpModal" class="help-modal">
                <div class="help-modal-content">
                    <span class="help-modal-close" onclick="closeAddUserHelp()">&times;</span>
                    <div class="help-content">
                        <h2>用户创建说明</h2>
                        
                        <h3>🎯 创建目的</h3>
                        <p>创建不同用户名的主要目的是为不同应用或用途提供独立的权限控制，确保每个应用只能访问其被授权的文件范围，避免单一应用权限过大带来的安全风险。</p>
                        
                        <h3>🔐 权限隔离原理</h3>
                        <p>每个用户都会被限制在特定的访问目录内，无法查看或修改其他用户的文件。这种设计实现了：</p>
                        <ul style="margin-left: 20px;">
                            <li><strong>应用隔离</strong>：不同应用使用不同用户，互不干扰</li>
                            <li><strong>权限最小化</strong>：每个用户只能访问必要文件</li>
                            <li><strong>风险管控</strong>：单个应用出现问题不会影响其他数据</li>
                        </ul>
                        
                        <h3>👥 给他人使用须知</h3>
                        <div class="warning">
                            <strong>⚠️ 隐私警告：</strong>虽然可以创建账户供他人登录使用，但请注意：
                            <ul style="margin-left: 20px; margin-top: 8px;">
                                <li><strong>文件非加密存储</strong>：所有文件以明文形式保存在服务器上</li>
                                <li><strong>管理员可见</strong>：作为系统管理员，您可以查看所有用户的文件内容</li>
                                <li><strong>隐私风险</strong>：他人可能未意识到文件对管理员可见</li>
                            </ul>
                        </div>
                        
                        <h3>💡 最佳实践建议</h3>
                        <p><strong>推荐使用场景：</strong></p>
                        <ul style="margin-left: 20px;">
                            <li>为不同应用创建专用账户（如：工作文档、个人备份、项目文件等）</li>
                            <li>临时文件传输（完成后及时删除账户）</li>
                            <li>团队协作（确保所有成员了解隐私政策）</li>
                        </ul>
                        
                        <p><strong>不建议使用场景：</strong></p>
                        <ul style="margin-left: 20px;">
                            <li>存储高度敏感的个人隐私文件</li>
                            <li>长期给他人使用而不告知隐私风险</li>
                            <li>存储需要绝对保密的商业机密</li>
                        </ul>
                    </div>
                </div>
            </div>
            
            <!-- WebDAV信息 -->
            <div id="webdav" class="tab-content">
                <h2>WebDAV访问信息</h2>
                <table>
                    <tr>
                        <th>服务器地址</th>
                        <td>
                            <a href="#" id="webdav-url-link" onclick="copyWebdavUrl(); return false;" 
                               style="color: #3498db; text-decoration: underline; cursor: pointer;" 
                               title="点击复制地址">
                                <?php echo 'http://' . $_SERVER['HTTP_HOST'] . rtrim(dirname($_SERVER['PHP_SELF']), '\/') . '/webdav.php'; ?>
                            </a>
                            <a href="<?php echo 'http://' . $_SERVER['HTTP_HOST'] . rtrim(dirname($_SERVER['PHP_SELF']), '\/') . '/webdav.php'; ?>" target="_blank" 
                               style="margin-left: 10px; color: #27ae60; text-decoration: none; font-size: 13px;">
                                访问
                            </a>
                        </td>
                    </tr>
                    <tr>
                        <th>用户名</th>
                        <td><?php echo htmlspecialchars($_SESSION['username']); ?></td>
                    </tr>
                    <tr>
                        <th>密码</th>
                        <td>您的登录密码</td>
                    </tr>
                </table>
                
                <p>您可以使用文件资源管理器、WebDAV客户端或其他支持WebDAV的应用程序连接到此服务。</p>
            </div>
            
            <script>
                // 密码保存功能
                function toggleSavePassword() {
                    const checkbox = document.getElementById('save_password');
                    const passwordInput = document.getElementById('add_password');
                    
                    if (checkbox.checked) {
                        // 保存密码到本地存储
                        localStorage.setItem('saved_password', passwordInput.value);
                    } else {
                        // 清除保存的密码
                        localStorage.removeItem('saved_password');
                    }
                }

                // 页面加载时恢复保存的密码
                document.addEventListener('DOMContentLoaded', function() {
                    const savedPassword = localStorage.getItem('saved_password');
                    if (savedPassword) {
                        document.getElementById('add_password').value = savedPassword;
                        document.getElementById('save_password').checked = true;
                    }
                });

                // 监听密码输入框变化，更新本地存储
                document.getElementById('add_password').addEventListener('input', function() {
                    const checkbox = document.getElementById('save_password');
                    if (checkbox.checked) {
                        localStorage.setItem('saved_password', this.value);
                    }
                });

                // 表单提交后，如果未勾选保存密码，则清除密码
                document.querySelector('form[method="post"]').addEventListener('submit', function(e) {
                    if (e.target.querySelector('input[name="action"]').value === 'add_user') {
                        const checkbox = document.getElementById('save_password');
                        if (!checkbox.checked) {
                            localStorage.removeItem('saved_password');
                            document.getElementById('add_password').value = '';
                        }
                    }
                });

                // 标签切换
                function switchTab(tabName) {
                    // 移除所有活动标签
                    document.querySelectorAll('.tab').forEach(function(t) {
                        t.classList.remove('active');
                    });
                    document.querySelectorAll('.tab-content').forEach(function(content) {
                        content.classList.remove('active');
                    });
                    
                    // 设置当前标签为活动
                    document.querySelector('.tab[data-tab="' + tabName + '"]').classList.add('active');
                    document.getElementById(tabName).classList.add('active');
                    
                    // 更新URL hash
                    window.location.hash = tabName;
                }

                // 标签点击事件
                document.querySelectorAll('.tab').forEach(function(tab) {
                    tab.addEventListener('click', function() {
                        switchTab(this.getAttribute('data-tab'));
                    });
                });

                // 页面加载时根据hash显示对应标签
                function loadTabFromHash() {
                    var hash = window.location.hash.substring(1);
                    var validTabs = ['account', 'users', 'webdav'];
                    
                    if (hash && validTabs.includes(hash)) {
                        switchTab(hash);
                    } else if (!hash) {
                        // 如果没有hash，默认显示第一个标签
                        switchTab('account');
                    }
                }

                // 监听hash变化
                window.addEventListener('hashchange', loadTabFromHash);
                
                // 页面加载时初始化
                document.addEventListener('DOMContentLoaded', loadTabFromHash);
                
                // 修改密码表单
                function showChangePasswordForm(userId, username) {
                    document.getElementById('change_user_id').value = userId;
                    document.getElementById('change_username').value = username;
                    document.getElementById('change_password_form').style.display = 'block';
                    document.getElementById('change_access_dir_form').style.display = 'none';
                    // 滚动到表单位置
                    document.getElementById('change_password_form').scrollIntoView({ behavior: 'smooth' });
                }
                
                function hideChangePasswordForm() {
                    document.getElementById('change_password_form').style.display = 'none';
                }

                // 修改访问目录表单
                function showChangeAccessDirForm(userId, username, currentDir) {
                    document.getElementById('change_access_dir_user_id').value = userId;
                    document.getElementById('change_access_dir_username').value = username;
                    document.getElementById('change_access_dir').value = currentDir;
                    document.getElementById('change_access_dir_form').style.display = 'block';
                    document.getElementById('change_password_form').style.display = 'none';
                    // 滚动到表单位置
                    document.getElementById('change_access_dir_form').scrollIntoView({ behavior: 'smooth' });
                }
                
                function hideChangeAccessDirForm() {
                    document.getElementById('change_access_dir_form').style.display = 'none';
                }
                
                // 修改访问目录确认提示
                function confirmChangeAccessDir() {
                    const username = document.getElementById('change_access_dir_username').value;
                    const newDir = document.getElementById('change_access_dir').value;
                    const defaultDir = username;
                    
                    const displayDir = newDir || defaultDir;
                    const message = `警告：修改过后将采用新目录 "${displayDir}"，是否确定要修改用户 ${username} 的访问目录？`;
                    
                    return confirm(message);
                }
                
                // 复制WebDAV地址
                function copyWebdavUrl() {
                    // 获取干净的URL（去除前后空格）
                    var url = document.getElementById('webdav-url-link').textContent.trim();
                    
                    // 创建toast提示元素
                    function showToast(message, duration = 2000) {
                        // 如果已存在toast，先移除
                        const existingToast = document.getElementById('copy-toast');
                        if (existingToast) {
                            existingToast.remove();
                        }
                        
                        const toast = document.createElement('div');
                        toast.id = 'copy-toast';
                        toast.style.cssText = `
                            position: fixed;
                            top: 20px;
                            right: 20px;
                            background-color: #27ae60;
                            color: white;
                            padding: 12px 24px;
                            border-radius: 4px;
                            box-shadow: 0 2px 8px rgba(0,0,0,0.2);
                            z-index: 10000;
                            font-size: 14px;
                            animation: slideIn 0.3s ease-out;
                        `;
                        toast.textContent = message;
                        
                        // 添加动画样式
                        const style = document.createElement('style');
                        style.textContent = `
                            @keyframes slideIn {
                                from { transform: translateX(100%); opacity: 0; }
                                to { transform: translateX(0); opacity: 1; }
                            }
                            @keyframes slideOut {
                                from { transform: translateX(0); opacity: 1; }
                                to { transform: translateX(100%); opacity: 0; }
                            }
                        `;
                        document.head.appendChild(style);
                        
                        document.body.appendChild(toast);
                        
                        // 2秒后移除
                        setTimeout(() => {
                            toast.style.animation = 'slideOut 0.3s ease-out';
                            setTimeout(() => {
                                toast.remove();
                                style.remove();
                            }, 300);
                        }, duration);
                    }
                    
                    if (navigator.clipboard) {
                        navigator.clipboard.writeText(url).then(function() {
                            showToast('复制成功');
                        }).catch(function(err) {
                            console.error('复制失败:', err);
                            showToast('复制失败，请重试');
                        });
                    } else {
                        // 使用备用方法
                        var textArea = document.createElement('textarea');
                        textArea.value = url;
                        textArea.style.position = 'fixed';
                        textArea.style.left = '-9999px';
                        textArea.style.top = '-9999px';
                        document.body.appendChild(textArea);
                        textArea.focus();
                        textArea.select();
                        
                        try {
                            var successful = document.execCommand('copy');
                            if (successful) {
                                showToast('复制成功');
                            } else {
                                showToast('复制失败，请手动复制');
                            }
                        } catch (err) {
                            console.error('复制失败:', err);
                            showToast('复制失败，请手动复制');
                        }
                        
                        document.body.removeChild(textArea);
                    }
                }
                
                // 添加用户帮助弹窗控制
                function showAddUserHelp() {
                    document.getElementById('addUserHelpModal').style.display = 'block';
                }
                
                function closeAddUserHelp() {
                    document.getElementById('addUserHelpModal').style.display = 'none';
                }
                
                // 点击弹窗外部关闭
                window.onclick = function(event) {
                    const modal = document.getElementById('addUserHelpModal');
                    if (event.target == modal) {
                        modal.style.display = 'none';
                    }
                }
                
                function fallbackCopyToClipboard(text) {
                    var textArea = document.createElement('textarea');
                    textArea.value = text;
                    textArea.style.position = 'fixed';
                    textArea.style.left = '-9999px';
                    textArea.style.top = '-9999px';
                    document.body.appendChild(textArea);
                    textArea.focus();
                    textArea.select();
                    
                    try {
                        var successful = document.execCommand('copy');
                        var msg = successful ? 'WebDAV地址已复制到剪贴板！' : '复制失败，请手动复制';
                        alert(msg);
                    } catch (err) {
                        console.error('复制失败:', err);
                        alert('复制失败，请手动复制');
                    }
                    
                    document.body.removeChild(textArea);
                }

                // 改变每页条数
                function changePerPage(perPage) {
                    const url = new URL(window.location.href);
                    url.searchParams.set('per_page', perPage);
                    url.searchParams.set('page', 1); // 重置到第一页
                    window.location.href = url.toString();
                }

                // 跳转到指定页
                function goToPage(page) {
                    const url = new URL(window.location.href);
                    url.searchParams.set('page', page);
                    window.location.href = url.toString();
                }
            </script>
        <?php endif; ?>
    </div>
</body>
</html>